<?php require_once '../../includes/header.php'; ?>

<div class="max-w-7xl mx-auto space-y-6 pb-10">

    <div class="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
        <div>
            <h1 class="text-3xl font-extrabold text-navy-700 tracking-tight">Dashboard</h1>
            <p class="text-sm text-gray-500 font-medium mt-1">Performance Analytics • 2024</p>
        </div>
        
        <div class="bg-white p-1.5 rounded-xl shadow-sm border border-gray-100 flex items-center">
            <button onclick="setScope('me')" id="btn-me" class="px-4 py-2 rounded-lg text-xs font-bold transition-all bg-navy-700 text-white shadow-md">
                My Stats
            </button>
            <button onclick="setScope('team')" id="btn-team" class="px-4 py-2 rounded-lg text-xs font-bold text-gray-500 hover:bg-gray-50 transition-all">
                Team View
            </button>
        </div>
    </div>

    <div id="dash-error" class="hidden bg-red-50 border-l-4 border-red-500 p-4 rounded-r-xl animate-pulse">
        <p class="text-sm text-red-700 font-bold">Sync Error</p>
        <p class="text-xs text-red-600 mt-1" id="dash-error-msg">Checking connection...</p>
    </div>

    <div class="grid grid-cols-2 md:grid-cols-4 gap-4 md:gap-6">
        
        <div class="relative overflow-hidden rounded-[24px] p-6 shadow-lg bg-gradient-to-br from-blue-500 to-blue-700 text-white group">
            <div class="absolute right-0 top-0 p-4 opacity-20 transform group-hover:scale-110 transition-transform duration-500">
                <i class="ph-bold ph-users-three text-6xl"></i>
            </div>
            <div class="relative z-10">
                <p class="text-xs font-bold text-blue-100 uppercase tracking-wider mb-1">Total Leads</p>
                <h3 class="text-4xl font-extrabold" id="kpi-total">--</h3>
                <div class="mt-4 inline-flex items-center gap-1 bg-white/20 px-2 py-1 rounded-lg backdrop-blur-sm">
                    <i class="ph-bold ph-calendar-blank text-xs"></i>
                    <span class="text-[10px] font-bold">2024 YTD</span>
                </div>
            </div>
        </div>

        <div class="relative overflow-hidden rounded-[24px] p-6 shadow-lg bg-gradient-to-br from-emerald-400 to-green-600 text-white group">
            <div class="absolute right-0 top-0 p-4 opacity-20 transform group-hover:scale-110 transition-transform duration-500">
                <i class="ph-bold ph-target text-6xl"></i>
            </div>
            <div class="relative z-10">
                <p class="text-xs font-bold text-green-100 uppercase tracking-wider mb-1">Qualified Rate</p>
                <h3 class="text-4xl font-extrabold" id="kpi-rate">--%</h3>
                <div class="mt-4 inline-flex items-center gap-1 bg-white/20 px-2 py-1 rounded-lg backdrop-blur-sm">
                    <span class="text-[10px] font-bold" id="kpi-won-count">0 Leads Won</span>
                </div>
            </div>
        </div>

        <div class="relative overflow-hidden rounded-[24px] p-6 shadow-lg bg-gradient-to-br from-orange-400 to-red-500 text-white group">
            <div class="absolute right-0 top-0 p-4 opacity-20 transform group-hover:scale-110 transition-transform duration-500">
                <i class="ph-bold ph-fire text-6xl"></i>
            </div>
            <div class="relative z-10">
                <p class="text-xs font-bold text-orange-100 uppercase tracking-wider mb-1">Hot Pipeline</p>
                <h3 class="text-4xl font-extrabold" id="kpi-hot">--</h3>
                <div class="mt-4 inline-flex items-center gap-1 bg-white/20 px-2 py-1 rounded-lg backdrop-blur-sm">
                    <i class="ph-bold ph-warning-circle text-xs"></i>
                    <span class="text-[10px] font-bold">Action Needed</span>
                </div>
            </div>
        </div>

        <div class="relative overflow-hidden rounded-[24px] p-6 shadow-lg bg-gradient-to-br from-indigo-500 to-purple-600 text-white group">
            <div class="absolute right-0 top-0 p-4 opacity-20 transform group-hover:scale-110 transition-transform duration-500">
                <i class="ph-bold ph-arrows-clockwise text-6xl"></i>
            </div>
            <div class="relative z-10">
                <p class="text-xs font-bold text-indigo-100 uppercase tracking-wider mb-1">Active</p>
                <h3 class="text-4xl font-extrabold" id="kpi-active">--</h3>
                <div class="mt-4 inline-flex items-center gap-1 bg-white/20 px-2 py-1 rounded-lg backdrop-blur-sm">
                    <span class="text-[10px] font-bold">In Progress</span>
                </div>
            </div>
        </div>
    </div>

    <div class="bg-white p-6 rounded-[24px] shadow-sm border border-gray-100">
        <div class="flex justify-between items-center mb-6">
            <div>
                <h4 class="text-xl font-bold text-navy-700">Growth Velocity</h4>
                <p class="text-sm text-gray-400">Leads created per month</p>
            </div>
            <div class="bg-blue-50 text-blue-600 px-3 py-1 rounded-lg text-xs font-bold">
                Line View
            </div>
        </div>
        <div class="h-72 w-full">
            <canvas id="lineChart"></canvas>
        </div>
    </div>

    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
        <div class="bg-white p-6 rounded-[24px] shadow-sm border border-gray-100 flex flex-col">
            <h4 class="text-lg font-bold text-navy-700 mb-2">Lead Status</h4>
            <div class="flex-1 min-h-[250px] relative">
                <canvas id="statusChart"></canvas>
            </div>
        </div>
        
        <div class="bg-white p-6 rounded-[24px] shadow-sm border border-gray-100 flex flex-col">
            <h4 class="text-lg font-bold text-navy-700 mb-2">Sources</h4>
            <div class="flex-1 min-h-[250px] relative flex justify-center">
                <canvas id="sourceChart"></canvas>
            </div>
        </div>
    </div>

</div>

<script>
    // --- CONFIG ---
    const API = "<?php echo CRM_URL; ?>api_gateway.php";
    const SESSION = localStorage.getItem('vt_session');
    const MY_ID = localStorage.getItem('vt_userid');
    
    // State
    let ALL_LEADS = []; // Holds fetched data
    let CHART_LINE = null;
    let CHART_STATUS = null;
    let CHART_SOURCE = null;
    let CURRENT_SCOPE = 'me'; // 'me' or 'team'

    document.addEventListener('DOMContentLoaded', fetchDashboardData);

    // --- 1. DATA FETCHING (Optimized) ---
    async function fetchDashboardData() {
        // Fetch up to 5000 leads for accuracy
        // We only need specific fields to keep it fast
        const q = "SELECT id, leadstatus, leadsource, assigned_user_id, createdtime FROM Leads WHERE createdtime >= '2024-01-01 00:00:00' LIMIT 5000;";
        
        try {
            const res = await fetch(`${API}?operation=query&sessionName=${SESSION}&query=${encodeURIComponent(q)}`);
            const data = await res.json();
            
            if (data.success) {
                ALL_LEADS = data.result;
                renderDashboard(); // Initial Render
            } else {
                showError("API Error: " + data.error.message);
            }
        } catch (error) {
            showError("Network Error. Check console.");
            console.error(error);
        }
    }

    // --- 2. FILTERING LOGIC ---
    function setScope(scope) {
        CURRENT_SCOPE = scope;
        
        // Update Buttons
        const btnMe = document.getElementById('btn-me');
        const btnTeam = document.getElementById('btn-team');
        
        if(scope === 'me') {
            btnMe.className = "px-4 py-2 rounded-lg text-xs font-bold transition-all bg-navy-700 text-white shadow-md";
            btnTeam.className = "px-4 py-2 rounded-lg text-xs font-bold text-gray-500 hover:bg-gray-50 transition-all";
        } else {
            btnTeam.className = "px-4 py-2 rounded-lg text-xs font-bold transition-all bg-navy-700 text-white shadow-md";
            btnMe.className = "px-4 py-2 rounded-lg text-xs font-bold text-gray-500 hover:bg-gray-50 transition-all";
        }

        renderDashboard();
    }

    function getFilteredLeads() {
        if (CURRENT_SCOPE === 'team') return ALL_LEADS;
        return ALL_LEADS.filter(l => l.assigned_user_id === MY_ID);
    }

    // --- 3. RENDER ENGINE ---
    function renderDashboard() {
        const leads = getFilteredLeads();
        
        // A. Process KPIs
        let hot = 0, qualified = 0, lost = 0, active = 0;
        const statusMap = {};
        const sourceMap = {};
        const monthMap = new Array(12).fill(0); // Jan-Dec

        leads.forEach(l => {
            const st = l.leadstatus || 'Unknown';
            const src = l.leadsource || 'Direct';
            
            // Month Calculation
            const d = new Date(l.createdtime);
            if(!isNaN(d)) monthMap[d.getMonth()]++;

            // Counters
            if(st === 'Hot') hot++;
            else if(st === 'Qualified') qualified++;
            else if(st.includes('Lost') || st.includes('Junk')) lost++;
            else active++; // New/Contacted/etc

            // Maps
            statusMap[st] = (statusMap[st] || 0) + 1;
            sourceMap[src] = (sourceMap[src] || 0) + 1;
        });

        const total = leads.length;
        
        // Update Cards
        animateValue("kpi-total", total);
        animateValue("kpi-hot", hot);
        animateValue("kpi-active", active);
        
        const rate = total > 0 ? Math.round((qualified/total)*100) : 0;
        document.getElementById('kpi-rate').innerText = rate + "%";
        document.getElementById('kpi-won-count').innerText = `${qualified} Qualified`;

        // Update Charts
        updateLineChart(monthMap);
        updateStatusChart(statusMap);
        updateSourceChart(sourceMap);
    }

    // --- 4. CHARTS ---
    
    // LINE CHART (Gradient Area)
    function updateLineChart(dataArr) {
        const ctx = document.getElementById('lineChart').getContext('2d');
        
        // Gradient Fill
        const gradient = ctx.createLinearGradient(0, 0, 0, 400);
        gradient.addColorStop(0, 'rgba(67, 24, 255, 0.4)');
        gradient.addColorStop(1, 'rgba(67, 24, 255, 0.0)');

        if(CHART_LINE) CHART_LINE.destroy();

        CHART_LINE = new Chart(ctx, {
            type: 'line',
            data: {
                labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
                datasets: [{
                    label: 'New Leads',
                    data: dataArr,
                    borderColor: '#4318FF',
                    backgroundColor: gradient,
                    borderWidth: 3,
                    pointBackgroundColor: '#fff',
                    pointBorderColor: '#4318FF',
                    pointRadius: 4,
                    fill: true,
                    tension: 0.4 // Smooth curve
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: { legend: { display: false } },
                scales: {
                    y: { beginAtZero: true, grid: { borderDash: [5, 5], color: '#f0f0f0' } },
                    x: { grid: { display: false } }
                }
            }
        });
    }

    function updateStatusChart(map) {
        const ctx = document.getElementById('statusChart');
        if(CHART_STATUS) CHART_STATUS.destroy();

        CHART_STATUS = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: Object.keys(map),
                datasets: [{
                    label: 'Count',
                    data: Object.values(map),
                    backgroundColor: [
                        '#4318FF', '#6AD2FF', '#EFF4FB', '#FFB547', '#E31A1A', '#10B981'
                    ],
                    borderRadius: 8,
                    barThickness: 30
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: { legend: { display: false } },
                scales: { x: { grid: { display: false } }, y: { display: false } }
            }
        });
    }

    function updateSourceChart(map) {
        const ctx = document.getElementById('sourceChart');
        if(CHART_SOURCE) CHART_SOURCE.destroy();

        CHART_SOURCE = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: Object.keys(map),
                datasets: [{
                    data: Object.values(map),
                    backgroundColor: ['#4318FF', '#3AB6E3', '#F59E0B', '#10B981', '#E31A1A'],
                    borderWidth: 0,
                    hoverOffset: 10
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                cutout: '75%',
                plugins: {
                    legend: { position: 'right', labels: { usePointStyle: true, boxWidth: 8, font: {size: 11} } }
                }
            }
        });
    }

    // --- UTILS ---
    function animateValue(id, end) {
        const obj = document.getElementById(id);
        if(!obj) return;
        let start = 0;
        const duration = 800;
        let startTimestamp = null;
        const step = (timestamp) => {
            if (!startTimestamp) startTimestamp = timestamp;
            const progress = Math.min((timestamp - startTimestamp) / duration, 1);
            obj.innerHTML = Math.floor(progress * (end - start) + start);
            if (progress < 1) window.requestAnimationFrame(step);
        };
        window.requestAnimationFrame(step);
    }

    function showError(msg) {
        document.getElementById('dash-error').classList.remove('hidden');
        document.getElementById('dash-error-msg').innerText = msg;
    }
</script>